unit MSSAX2View;

{
  A viewer for XML documents.
  Based on SAX2 interfaces, using Microsoft SAX2 parser.
  Requires MSXML v3 package from Microsoft.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written 11 June, 2000.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Menus, ComCtrls, Grids, TypInfo,
{$IFDEF VER120}  { Delphi 4 }
  ImgList,
{$ENDIF}
{$IFDEF VER140}  { Delphi 6 }
  Variants,
{$ENDIF}
{$IFDEF VER150}  { Delphi 7 }
  Variants,
{$ENDIF}
  ActiveX, MSXML2_tlb;

type
  TfrmSAX2Viewer = class(TForm, IVBSAXEntityResolver, IVBSAXDTDHandler,
      IVBSAXContentHandler, IVBSAXErrorHandler,
      IVBSAXLexicalHandler, IVBSAXDeclHandler)
    pgcMain: TPageControl;
      tshStructure: TTabSheet;
        trvXML: TTreeView;
        pgcDetails: TPageControl;
          tshDocument: TTabSheet;
            Label1: TLabel;
            edtDocType: TEdit;
            Label2: TLabel;
            edtPublicId: TEdit;
            Label3: TLabel;
            edtSystemId: TEdit;
            Label6: TLabel;
            stgEntities: TStringGrid;
            Label7: TLabel;
            stgNotations: TStringGrid;
          tshElement: TTabSheet;
            pnlNames: TPanel;
              Label4: TLabel;
              edtURI: TEdit;
              Label5: TLabel;
              edtLocalName: TEdit;
            stgAttributes: TStringGrid;
            stgPrefixes: TStringGrid;
          tshText: TTabSheet;
            lblNodeType: TLabel;
            memText: TMemo;
      tshSource: TTabSheet;
        memSource: TRichEdit;
    mnuMain: TMainMenu;
      mniFile: TMenuItem;
        mniOpen: TMenuItem;
        mniSep1: TMenuItem;
        mniParserOptions: TMenuItem;
          mniValidation: TMenuItem;
          mniNamespaces: TMenuItem;
          mniNamespacePrefixes: TMenuItem;
        mniSep2: TMenuItem;
        mniExit: TMenuItem;
      mniView: TMenuItem;
        mniExpandAll: TMenuItem;
        mniCollapseAll: TMenuItem;
        mniSep3: TMenuItem;
        mniViewSource: TMenuItem;
    imlXML: TImageList;
    dlgOpen: TOpenDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure mniOpenClick(Sender: TObject);
    procedure mniParserOptionClick(Sender: TObject);
    procedure mniExitClick(Sender: TObject);
    procedure mniExpandAllClick(Sender: TObject);
    procedure mniCollapseAllClick(Sender: TObject);
    procedure mniViewSourceClick(Sender: TObject);
    procedure trvXMLChange(Sender: TObject; Node: TTreeNode);
  private
    FCharIcon: Integer;
    FCurrent: TTreeNode;
    FLocator: IVBSAXLocator;
    FPrefixes: TStringList;
    FSAXReader: IVBSAXXMLReader;
    procedure ClearTree;
    procedure LoadDoc(Filename: string);
    procedure ShowError(Level: TMsgDlgType; const oLocator: IVBSAXLocator;
      const strError: WideString; nErrorCode: Integer);
    function TruncateText(Text: string): string;
  public
    { IVBSAXEntityResolver }
    function resolveEntity(var strPublicId: WideString;
      var strSystemId: WideString): OleVariant; safecall;
    { IVBSAXDTDHandler }
    procedure notationDecl(var strName: WideString;
      var strPublicId: WideString; var strSystemId: WideString); safecall;
    procedure unparsedEntityDecl(var strName: WideString;
      var strPublicId: WideString; var strSystemId: WideString;
      var strNotationName: WideString); safecall;
    { IVBSAXContentHandler }
    procedure _Set_documentLocator(const Param1: IVBSAXLocator); safecall;
    procedure startDocument; safecall;
    procedure endDocument; safecall;
    procedure startPrefixMapping(var strPrefix: WideString;
      var strURI: WideString); safecall;
    procedure endPrefixMapping(var strPrefix: WideString); safecall;
    procedure startElement(var strNamespaceURI: WideString;
      var strLocalName: WideString; var strQName: WideString;
      const oAttributes: IVBSAXAttributes); safecall;
    procedure endElement(var strNamespaceURI: WideString;
      var strLocalName: WideString; var strQName: WideString); safecall;
    procedure characters(var strChars: WideString); safecall;
    procedure ignorableWhitespace(var strChars: WideString); safecall;
    procedure processingInstruction(var strTarget: WideString;
      var strData: WideString); safecall;
    procedure skippedEntity(var strName: WideString); safecall;
    { IVBSAXLexicalHandler }
    procedure startDTD(var strName: WideString; var strPublicId: WideString;
      var strSystemId: WideString); safecall;
    procedure endDTD; safecall;
    procedure startEntity(var strName: WideString); safecall;
    procedure endEntity(var strName: WideString); safecall;
    procedure startCDATA; safecall;
    procedure endCDATA; safecall;
    procedure comment(var strChars: WideString); safecall;
    { IVBSAXDeclHandler }
    procedure elementDecl(var strName: WideString;
      var strModel: WideString); safecall;
    procedure attributeDecl(var strElementName: WideString;
      var strAttributeName: WideString; var strType: WideString;
      var strValueDefault: WideString; var strValue: WideString); safecall;
    procedure internalEntityDecl(var strName: WideString;
      var strValue: WideString); safecall;
    procedure externalEntityDecl(var strName: WideString;
      var strPublicId: WideString; var strSystemId: WideString); safecall;
    { IVBSAXErrorHandler }
    procedure error(const oLocator: IVBSAXLocator; var strError: WideString;
      nErrorCode: Integer); safecall;
    procedure fatalError(const oLocator: IVBSAXLocator;
      var strError: WideString; nErrorCode: Integer); safecall;
    procedure ignorableWarning(const oLocator: IVBSAXLocator;
      var strError: WideString; nErrorCode: Integer); safecall;
  end;

var
  frmSAX2Viewer: TfrmSAX2Viewer;

implementation

{$R *.DFM}

resourcestring
  AttributeDeclDesc = 'Attribute Declaration';
  AttributeDecln    = '%s %s %s';
  AttributeDesc     = 'Attribute';
  CDataDesc         = 'CDATA Section';
  CommentDesc       = 'Comment';
  DTDDeclDesc       = 'DTD Declaration';
  DTDDecln          = '%s PUBLIC "%s" SYSTEM "%s"';
  DTDDesc           = 'DTD';
  ElementDeclDesc   = 'Element Declaration';
  EntityDecln       = 'PUBLIC "%s" SYSTEM "%s"';
  EntityDesc        = 'Entity Declaration/Reference';
  ErrorLocation     = '(%d) %s'#13#10'at line %d, column %d';
  InstructionDesc   = 'Processing Instruction';
  NameDesc          = 'Name';
  NotationDesc      = 'Notation';
  PrefixDesc        = 'Prefix';
  PublicDesc        = 'Public Id';
  SkippedDesc       = 'Skipped entity';
  SystemDesc        = 'System Id';
  TextDesc          = 'Text';
  URIDesc           = 'URI';
  ValueDesc         = 'Value';

const
  { Standard feature and property names }
  NamespacesFeature: WideString        =
    'http://xml.org/sax/features/namespaces';
  NamespacePrefixesFeature: WideString =
    'http://xml.org/sax/features/namespace-prefixes';
  ValidationFeature: WideString        =
    'http://xml.org/sax/features/validation';
  DeclHandlerProperty: WideString      =
    'http://xml.org/sax/properties/declaration-handler';
  LexicalHandlerProperty: WideString   =
    'http://xml.org/sax/properties/lexical-handler';

  { Icons for tree view }
  DocumentIcon      = 0;
  ElementIcon       = 1;
  ProcInstrIcon     = 2;
  TextIcon          = 3;
  CDataIcon         = 4;
  CommentIcon       = 5;
  DTDIcon           = 6;
  ElementDeclIcon   = 7;
  AttributeDeclIcon = 8;
  EntityIcon        = 9;

{ TString ---------------------------------------------------------------------}

type
  { Wrapper around a string }
  TString = class(TObject)
  private
    FValue: string;
  public
    constructor Create(Value: string);
    property Value: string read FValue write FValue;
  end;

{ Initialisation }
constructor TString.Create(Value: string);
begin
  inherited Create;
  FValue := Value;
end;

{ TElement --------------------------------------------------------------------}

type
  { Details about an element }
  TElement = class(TObject)
  private
    FAttributes: TStringList;
    FLocalName: string;
    FPrefixes: TStringList;
    FURI: string;
  public
    constructor Create(URI, LocalName: string);
    destructor Destroy; override;
    property Attributes: TStringList read FAttributes write FAttributes;
    property LocalName: string read FLocalName write FLocalName;
    property Prefixes: TStringList read FPrefixes write FPrefixes;
    property URI: string read FURI write FURI;
  end;

{ Initialisation }
constructor TElement.Create(URI, LocalName: string);
begin
  inherited Create;
  FURI        := URI;
  FLocalName  := LocalName;
  FAttributes := TStringList.Create;
  FPrefixes   := TStringList.Create;
end;

{ Release resources }
destructor TElement.Destroy;
begin
  FAttributes.Free;
  FPrefixes.Free;
  inherited Destroy;
end;

{ TfrmSAX2Viewer --------------------------------------------------------------}

{ Initialisation - load the XML document on start up }
procedure TfrmSAX2Viewer.FormCreate(Sender: TObject);
begin
  FPrefixes          := TStringList.Create;
  dlgOpen.InitialDir := ExtractFilePath(Application.ExeName);
  with stgEntities do
  begin
    Cells[0, 0] := NameDesc;
    Cells[1, 0] := PublicDesc;
    Cells[2, 0] := SystemDesc;
    Cells[3, 0] := NotationDesc;
  end;
  with stgNotations do
  begin
    Cells[0, 0] := NameDesc;
    Cells[1, 0] := PublicDesc;
    Cells[2, 0] := SystemDesc;
  end;
  with stgAttributes do
  begin
    Cells[0, 0] := AttributeDesc;
    Cells[1, 0] := ValueDesc;
  end;
  with stgPrefixes do
  begin
    Cells[0, 0] := PrefixDesc;
    Cells[1, 0] := URIDesc;
  end;
  pgcDetails.ActivePage := tshDocument;
  { Load XML reader }
  FSAXReader := CoSAXXMLReader.Create;
  { Set standard handlers }
  FSAXReader._Set_contentHandler(Self);
  FSAXReader._Set_dtdHandler(Self);
  FSAXReader._Set_errorHandler(Self);
  { Currently not implemented }
//  FSAXReader._Set_entityResolver(Self);
  { Set extension handlers }
  try
    FSAXReader.putProperty(LexicalHandlerProperty, IVBSAXLexicalHandler(Self));
  except
    { Ignore }
  end;
  try
    FSAXReader.putProperty(DeclHandlerProperty, IVBSAXDeclHandler(Self));
  except
    { Ignore }
  end;
end;

{ Release resources }
procedure TfrmSAX2Viewer.FormDestroy(Sender: TObject);
begin
  ClearTree;
  FSAXReader := nil;
  FLocator   := nil;
  FPrefixes.Free;
end;

{ Empty the tree of associated objects }
procedure TfrmSAX2Viewer.ClearTree;
var
  Index: Integer;
begin
  for Index := 0 to trvXML.Items.Count - 1 do
    if Assigned(trvXML.Items[Index].Data) then
      TObject(trvXML.Items[Index].Data).Free;
  trvXML.OnChange := nil;
  trvXML.Items.Clear;
  trvXML.OnChange := trvXMLChange;
end;

{ Load an XML document }
procedure TfrmSAX2Viewer.LoadDoc(Filename: string);
begin
  Screen.Cursor := crHourGlass;
  try
    trvXML.Items.BeginUpdate;
    try
      pgcDetails.ActivePage := tshDocument;
      { Load the source document }
      memSource.Lines.LoadFromFile(Filename);
      dlgOpen.Filename      := Filename;
      { Attempt to set standard features from menu items }
      try
        FSAXReader.putFeature(ValidationFeature, mniValidation.Checked);
      except
        mniValidation.Enabled := False;
      end;
      try
        FSAXReader.putFeature(NamespacesFeature, mniNamespaces.Checked);
      except
        mniNamespaces.Enabled := False;
      end;
      try
        FSAXReader.putFeature(NamespacePrefixesFeature,
          mniNamespacePrefixes.Checked);
      except
        mniNamespacePrefixes.Enabled := False;
      end;
      { Parse the document - form already registered with parser as handlers }
      FSAXReader.parseURL(Filename);
    finally
      trvXML.Items.EndUpdate;
    end;
  finally
    Screen.Cursor := crDefault;
  end;
end;

{ Select a file to open }
procedure TfrmSAX2Viewer.mniOpenClick(Sender: TObject);
begin
  with dlgOpen do
    if Execute then
      LoadDoc(Filename);
end;

{ Toggle the parser options }
procedure TfrmSAX2Viewer.mniParserOptionClick(Sender: TObject);
begin
  with TMenuItem(Sender) do
    Checked := not Checked;
end;

{ Exit the application }
procedure TfrmSAX2Viewer.mniExitClick(Sender: TObject);
begin
  Close;
end;

{ Expand all nodes below the current one }
procedure TfrmSAX2Viewer.mniExpandAllClick(Sender: TObject);
begin
  if Assigned(trvXML.Selected) then
    trvXML.Selected.Expand(True);
end;

{ Collapse all nodes below the current one }
procedure TfrmSAX2Viewer.mniCollapseAllClick(Sender: TObject);
begin
  if Assigned(trvXML.Selected) then
    trvXML.Selected.Collapse(True);
end;

{ Toggle between structure and source }
procedure TfrmSAX2Viewer.mniViewSourceClick(Sender: TObject);
begin
  mniViewSource.Checked := not mniViewSource.Checked;
  if mniViewSource.Checked then
    pgcMain.ActivePage := tshSource
  else
    pgcMain.ActivePage := tshStructure;
end;

{ Show an error message and position cursor in source text }
procedure TfrmSAX2Viewer.ShowError(Level: TMsgDlgType;
  const oLocator: IVBSAXLocator; const strError: WideString;
  nErrorCode: Integer);
var
  Line, SelStart: Integer;
  XMLSource: string;
begin
  MessageDlg(Format(ErrorLocation, [nErrorCode, strError,
    oLocator.lineNumber, oLocator.columnNumber]), Level, [mbOK], 0);
  mniViewSource.Checked := True;
  pgcMain.ActivePage    := tshSource;
  SelStart              := 0;
  XMLSource             := memSource.Lines.Text;
  for Line := 1 to oLocator.lineNumber - 1 do
  begin
    SelStart := SelStart + Pos(#10, XMLSource);
    Delete(XMLSource, 1, Pos(#10, XMLSource));
  end;
  memSource.SelStart  := SelStart + oLocator.columnNumber - 1;
  memSource.SelLength := 0;
end;

{ Truncate text to a given length }
function TfrmSAX2Viewer.TruncateText(Text: string): string;
begin
  if Length(Text) > 20 then
    Result := Copy(Text, 1, 17) + '...'
  else
    Result := Text;
end;

{ Display details for the selected XML element }
procedure TfrmSAX2Viewer.trvXMLChange(Sender: TObject; Node: TTreeNode);
var
  Index: Integer;
  Element: TElement;
begin
  if not Assigned(trvXML.Selected.Data) then
    { Document }
    pgcDetails.ActivePage := tshDocument
  else if TObject(trvXML.Selected.Data) is TString then
  begin
    { Text/processing instruction/etc }
    pgcDetails.ActivePage := tshText;
    case trvXML.Selected.ImageIndex of
      AttributeDeclIcon: lblNodeType.Caption := AttributeDeclDesc;
      CDataIcon:         lblNodeType.Caption := CDataDesc;
      CommentIcon:       lblNodeType.Caption := CommentDesc;
      DTDIcon:           lblNodeType.Caption := DTDDeclDesc;
      ElementDeclIcon:   lblNodeType.Caption := ElementDeclDesc;
      EntityIcon:        lblNodeType.Caption := EntityDesc;
      ProcInstrIcon:     lblNodeType.Caption := InstructionDesc;
      else               lblNodeType.Caption := TextDesc;
    end;
    memText.Lines.Text := TString(trvXML.Selected.Data).Value;
  end
  else if TObject(trvXML.Selected.Data) is TElement then
  begin
    { Element }
    pgcDetails.ActivePage := tshElement;
    Element               := TElement(trvXML.Selected.Data);
    edtURI.Text           := Element.URI;
    edtLocalName.Text     := Element.LocalName;
    with stgAttributes do
    begin
      if Element.Attributes.Count = 0 then
        RowCount := 2
      else
        RowCount := Element.Attributes.Count + 1;
      Rows[1].Clear;
      for Index := 0 to Element.Attributes.Count - 1 do
      begin
        Cells[0, Index + 1] := Element.Attributes.Names[Index];
        Cells[1, Index + 1] :=
          Element.Attributes.Values[Element.Attributes.Names[Index]];
      end;
    end;
    with stgPrefixes do
    begin
      if Element.Prefixes.Count = 0 then
        RowCount := 2
      else
        RowCount := Element.Prefixes.Count + 1;
      Rows[1].Clear;
      for Index := 0 to Element.Prefixes.Count - 1 do
      begin
        Cells[0, Index + 1] := Element.Prefixes.Names[Index];
        Cells[1, Index + 1] :=
          Element.Prefixes.Values[Element.Prefixes.Names[Index]];
      end;
    end;
  end;
end;

{ SAX handler implementations -------------------------------------------------}

{ IVBSAXEntityResolver --------------------------------------------------------}

function TfrmSAX2Viewer.resolveEntity(var strPublicId,
  strSystemId: WideString): OleVariant;
begin
  Result := Null;
end;

{ IVBSAXDTDHandler ------------------------------------------------------------}

{ Save the details about a notation }
procedure TfrmSAX2Viewer.notationDecl(
  var strName, strPublicId, strSystemId: WideString);
begin
  with stgNotations do
  begin
    if Cells[0, 1] <> '' then
      RowCount := RowCount + 1;
    Cells[0, RowCount - 1] := strName;
    Cells[1, RowCount - 1] := strPublicId;
    Cells[2, RowCount - 1] := strSystemId;
  end;
end;

{ Save the details about an external entity }
procedure TfrmSAX2Viewer.unparsedEntityDecl(
  var strName, strPublicId, strSystemId, strNotationName: WideString);
begin
  with stgEntities do
  begin
    if Cells[0, 1] <> '' then
      RowCount := RowCount + 1;
    Cells[0, RowCount - 1] := strName;
    Cells[1, RowCount - 1] := strPublicId;
    Cells[2, RowCount - 1] := strSystemId;
    Cells[3, RowCount - 1] := strNotationName;
  end;
end;

{ IVBSAXContentHandler --------------------------------------------------------}

{ Add a text node to the tree }
procedure TfrmSAX2Viewer.characters(var strChars: WideString);
var
  Index: Integer;
  Text: string;
begin
  { Ignore all white space }
  Text := strChars;
  for Index := 1 to Length(Text) do
    if Text[Index] > ' ' then
      Break;
  if Index > Length(Text) then
    Exit;

  with trvXML.Items.AddChildObject(FCurrent, TruncateText(strChars),
    TString.Create(strChars)) do
  begin
    ImageIndex    := FCharIcon;
    SelectedIndex := FCharIcon;
  end;
end;

{ Tidy up and expand the top level of the tree }
procedure TfrmSAX2Viewer.endDocument;
begin
  trvXML.Items[0].Expand(False);
end;

{ Move the current context up the hierarchy when an element ends }
procedure TfrmSAX2Viewer.endElement(
  var strNamespaceURI, strLocalName, strQName: WideString);
begin
  FCurrent := FCurrent.Parent;
end;

{ Note end of prefix mapping }
procedure TfrmSAX2Viewer.endPrefixMapping(var strPrefix: WideString);
begin
  { Do nothing }
end;

{ As the name says - ignore this }
procedure TfrmSAX2Viewer.ignorableWhitespace(var strChars: WideString);
begin
  { Do nothing }
end;

{ Add a processing instruction to the tree }
procedure TfrmSAX2Viewer.processingInstruction(
  var strTarget, strData: WideString);
begin
  with trvXML.Items.AddChildObject(
    FCurrent, strTarget, TString.Create(strData)) do
  begin
    ImageIndex    := ProcInstrIcon;
    SelectedIndex := ProcInstrIcon;
  end;
end;

{ Save the locator for later }
procedure TfrmSAX2Viewer._Set_documentLocator(const Param1: IVBSAXLocator);
begin
  FLocator := Param1;
end;

{ Add a skipped entity to the tree }
procedure TfrmSAX2Viewer.skippedEntity(var strName: WideString);
begin
  with trvXML.Items.AddChildObject(
    FCurrent, strName, TString.Create(SkippedDesc)) do
  begin
    ImageIndex    := EntityIcon;
    SelectedIndex := EntityIcon;
  end;
end;

{ Initialisation for a new document display }
procedure TfrmSAX2Viewer.startDocument;
begin
  ClearTree;
  FCharIcon              := TextIcon;
  FCurrent               := trvXML.Items.AddChild(nil, dlgOpen.FileName);
  FCurrent.ImageIndex    := DocumentIcon;
  FCurrent.SelectedIndex := DocumentIcon;
  edtDocType.Text        := '';
  edtPublicId.Text       := '';
  edtSystemId.Text       := dlgOpen.FileName;
  stgEntities.RowCount   := 2;
  stgEntities.Rows[1].Clear;
  stgNotations.RowCount  := 2;
  stgNotations.Rows[1].Clear;
end;

{ Note this element as the current node and save its attributes }
procedure TfrmSAX2Viewer.startElement(
  var strNamespaceURI, strLocalName, strQName: WideString;
  const oAttributes: IVBSAXAttributes);
var
  Element: TElement;
  Index: Integer;
begin
  Element := TElement.Create(strNamespaceUri, strLocalName);
  for Index := 0 to oAttributes.Length - 1 do
    Element.Attributes.Values[oAttributes.getQName(Index)] :=
      oAttributes.getValue(Index);
  Element.Prefixes.Assign(FPrefixes);
  FPrefixes.Clear;
  FCurrent               :=
    trvXML.Items.AddChildObject(FCurrent, strQName, Element);
  FCurrent.ImageIndex    := ElementIcon;
  FCurrent.SelectedIndex := ElementIcon;
  if edtDocType.Text = '' then
    edtDocType.Text := strQName;
end;

{ Save prefix for display with element }
procedure TfrmSAX2Viewer.startPrefixMapping(var strPrefix, strURI: WideString);
begin
  FPrefixes.Values[strPrefix] := strUri;
end;

{ IVBSAXDeclHandler -----------------------------------------------------------}

{ Add a comment to the tree }
procedure TfrmSAX2Viewer.comment(var strChars: WideString);
begin
  with trvXML.Items.AddChildObject(
    FCurrent, TruncateText(strChars), TString.Create(strChars)) do
  begin
    ImageIndex    := CommentIcon;
    SelectedIndex := CommentIcon;
  end;
end;

{ Note end of CDATA section }
procedure TfrmSAX2Viewer.endCDATA;
begin
  FCharIcon := TextIcon;
end;

{ Move the current context up the hierarchy when the DTD ends }
procedure TfrmSAX2Viewer.endDTD;
begin
  FCurrent := FCurrent.Parent;
end;

{ Move the current context up the hierarchy when an entity reference ends }
procedure TfrmSAX2Viewer.endEntity(var strName: WideString);
begin
  FCurrent := FCurrent.Parent;
end;

{ Note start of CDATA section - text returned through characters method }
procedure TfrmSAX2Viewer.startCDATA;
begin
  FCharIcon := CDataIcon;
end;

{ Add a DTD declaration to the tree }
procedure TfrmSAX2Viewer.startDTD(var strName, strPublicId,
  strSystemId: WideString);
begin
  FCurrent := trvXML.Items.AddChildObject(FCurrent, DTDDesc,
    TString.Create(Format(DTDDecln, [strName, strPublicId, strSystemId])));
  with FCurrent do
  begin
    ImageIndex    := DTDIcon;
    SelectedIndex := DTDIcon;
  end;
end;

{ Add an entity reference to the tree }
procedure TfrmSAX2Viewer.startEntity(var strName: WideString);
begin
  FCurrent := trvXML.Items.AddChildObject(
    FCurrent, strName, TString.Create(strName));
  with FCurrent do
  begin
    ImageIndex    := EntityIcon;
    SelectedIndex := EntityIcon;
  end;
end;

{ IVBSAXDeclHandler -----------------------------------------------------------}

{ Add an attribute declaration to the tree }
procedure TfrmSAX2Viewer.attributeDecl(var strElementName,
  strAttributeName, strType, strValueDefault, strValue: WideString);
begin
  with trvXML.Items.AddChildObject(FCurrent,
    strElementName + '.' + strAttributeName, TString.Create(
    Format(AttributeDecln, [strType, strValueDefault, strValue]))) do
  begin
    ImageIndex    := AttributeDeclIcon;
    SelectedIndex := AttributeDeclIcon;
  end;
end;

{ Add an element declaration to the tree }
procedure TfrmSAX2Viewer.elementDecl(var strName, strModel: WideString);
begin
  with trvXML.Items.AddChildObject(
    FCurrent, strName, TString.Create(strModel)) do
  begin
    ImageIndex    := ElementDeclIcon;
    SelectedIndex := ElementDeclIcon;
  end;
end;

{ Add an entity declaration to the tree }
procedure TfrmSAX2Viewer.externalEntityDecl(var strName, strPublicId,
  strSystemId: WideString);
begin
  with trvXML.Items.AddChildObject(FCurrent, strName,
    TString.Create(Format(EntityDecln, [strPublicId, strSystemId]))) do
  begin
    ImageIndex    := EntityIcon;
    SelectedIndex := EntityIcon;
  end;
end;

{ Add an entity declaration to the tree }
procedure TfrmSAX2Viewer.internalEntityDecl(var strName, strValue: WideString);
begin
  with trvXML.Items.AddChildObject(
    FCurrent, strName, TString.Create(strValue)) do
  begin
    ImageIndex    := EntityIcon;
    SelectedIndex := EntityIcon;
  end;
end;

{ IVBSAXErrorHandler ----------------------------------------------------------}

{ Notify the user of the error }
procedure TfrmSAX2Viewer.error(const oLocator: IVBSAXLocator;
  var strError: WideString; nErrorCode: Integer);
begin
  ShowError(mtError, oLocator, strError, nErrorCode);
end;

{ Notify the user of the error }
procedure TfrmSAX2Viewer.fatalError(const oLocator: IVBSAXLocator;
  var strError: WideString; nErrorCode: Integer);
begin
  ShowError(mtError, oLocator, strError, nErrorCode);
end;

{ Notify the user of the warning }
procedure TfrmSAX2Viewer.ignorableWarning(const oLocator: IVBSAXLocator;
  var strError: WideString; nErrorCode: Integer);
begin
  ShowError(mtWarning, oLocator, strError, nErrorCode);
end;

end.
